
let panel=null;
function getSelectionOrFocusedText(){
  const sel=window.getSelection?.();
  const selected = sel && sel.toString ? sel.toString().trim() : '';
  const active=document.activeElement;
  if(active && (active.tagName==='TEXTAREA' || (active.tagName==='INPUT' && (active.type==='text' || active.type==='search')))){
    const v=active.value || '';
    const start=active.selectionStart ?? 0;
    const end=active.selectionEnd ?? 0;
    const chunk = (start!==end) ? v.slice(start,end).trim() : v.trim();
    return {text: selected || chunk, target: active};
  }
  if(active && active.isContentEditable){
    const chunk=(active.innerText||'').trim();
    return {text: selected || chunk, target: active};
  }
  return {text:selected, target:null};
}
function ensurePanel(){
  if(panel) return panel;
  panel=document.createElement('div');
  panel.className='mz-guard-panel';
  panel.innerHTML = `
    <div class="mz-guard-head">
      <div class="mz-guard-title">MyLaw.Zone Writing Guardian</div>
      <button class="mz-guard-close" type="button">Close</button>
    </div>
    <div class="mz-guard-body">
      <textarea id="mz_in" placeholder="Paste or select text to improve..."></textarea>
      <div class="mz-guard-actions">
        <button class="primary" id="mz_guard" type="button">Guard</button>
        <button id="mz_apply" type="button">Apply</button>
      </div>
      <div class="mz-guard-out" id="mz_out">Suggestions will appear here.</div>
      <div class="mz-guard-note">General guidance only. For sensitive/legal matters, use MyLaw.Zone to Start a Case or consult a qualified lawyer.</div>
    </div>`;
  document.documentElement.appendChild(panel);
  panel.querySelector('.mz-guard-close').addEventListener('click',()=>{panel.remove(); panel=null;});
  return panel;
}
async function getConfig(){
  return new Promise(resolve=>{
    chrome.storage.sync.get({mzBaseUrl:'https://mylaw.zone', mzToken:''}, v=>resolve({base:(v.mzBaseUrl||'https://mylaw.zone').replace(/\/$/,''), token:(v.mzToken||'')}));
  });
}
async function callAI(text){
  const cfg=await getConfig();
  const base=cfg.base;
  const token=cfg.token;
  if(!token){ throw new Error("No Extension Token set. Open extension Options and paste your token from /extension-token."); }
  const url=base + '/wp-json/mylawzone/v1/ai/extension';
  const prompt = `Improve the writing for clarity, professionalism, and safety. Keep the same meaning. If it could be defamatory or risky, suggest a safer phrasing.\n\nTEXT:\n${text}`;
  const res=await fetch(url,{
    method:'POST',
    headers:{'Content-Type':'application/json','X-MyLawZone-Token': token},
    body: JSON.stringify({prompt})
  });
  if(!res.ok){
    const t=await res.text().catch(()=> '');
    throw new Error('Request failed: '+res.status+' '+t.slice(0,160));
  }
  const json=await res.json();
  return json.answer || '';
}
let lastTarget=null;
async function guardOnce(){
  const {text,target}=getSelectionOrFocusedText();
  const p=ensurePanel();
  if(text) p.querySelector('#mz_in').value=text;
  lastTarget=target;
  p.querySelector('#mz_out').textContent='Working...';
  try{
    const ans=await callAI(p.querySelector('#mz_in').value.trim());
    p.querySelector('#mz_out').textContent=ans || 'No response.';
  }catch(e){
    p.querySelector('#mz_out').textContent=String(e.message||e);
  }
}
function applyToTarget(){
  if(!panel) return;
  const out=panel.querySelector('#mz_out').textContent;
  if(!out || out==='Working...' || out==='Suggestions will appear here.') return;
  if(lastTarget){
    if(lastTarget.tagName==='TEXTAREA' || lastTarget.tagName==='INPUT'){
      lastTarget.value = out;
      lastTarget.dispatchEvent(new Event('input',{bubbles:true}));
    }else if(lastTarget.isContentEditable){
      lastTarget.innerText = out;
      lastTarget.dispatchEvent(new Event('input',{bubbles:true}));
    }
  }
}
chrome.runtime.onMessage.addListener((msg)=>{
  if(msg?.type==='MZ_TOGGLE_SIDEBAR'){
    if(panel){ panel.remove(); panel=null; }
    else ensurePanel();
  }
  if(msg?.type==='MZ_GUARD_ONCE'){ guardOnce(); }
});
document.addEventListener('click', (e)=>{
  if(e.target && e.target.id==='mz_guard'){ guardOnce(); }
  if(e.target && e.target.id==='mz_apply'){ applyToTarget(); }
});
